using System;
using System.Runtime.Serialization;

namespace Team_Project.Exceptions
{
	/// <summary>
	/// Eccezione utilizzata per segnalare l'accesso ad una location inesistente
	/// </summary>
	[Serializable]
	public class LocationNotExistsException : TeamProjectException, ISerializable
	{
		/// <summary>
		/// Contiene il percorso della location
		/// </summary>
		protected string location;

		/// <summary>
		/// Costruttore necessario per la serializzazione
		/// </summary>
		/// <param name="info">info</param>
		/// <param name="context">context</param>
		protected LocationNotExistsException(SerializationInfo info, StreamingContext context):
		base(info,context)
		{
			location = (string)info.GetValue("location",typeof(string));
		}

		/// <summary>
		/// Costruisce un'istanza dell'eccezione
		/// </summary>
		public LocationNotExistsException():base()
		{}

		/// <summary>
		/// Costruisce un'istanza dell'eccezione
		/// </summary>
		/// <param name="location">Percorso della location alla quale si tentava di
		/// accedere</param>
		public LocationNotExistsException(string location):
		this(location,location + " does not exists")
		{}

		/// <summary>
		/// Costruisce un'istanza dell'eccezione
		/// </summary>
		/// <param name="location">Percorso della location alla quale si tentava di
		/// accedere</param>
		/// <param name="message">Messaggio di errore aggiuntivo</param>
		public LocationNotExistsException(string location, string message):
		this(location,message,null)
		{}

		/// <summary>
		/// Costruisce un'istanza dell'eccezione
		/// </summary>
		/// <param name="location">Percorso della location alla quale si tentava di
		/// accedere</param>
		/// <param name="message">Messaggio di errore aggiuntivo</param>
		/// <param name="inner">Eccezione interna</param>
		public LocationNotExistsException(string location, string message, Exception inner)
			:base(message,inner)
		{
			this.location = location;
		}

		/// <summary>
		/// Ottiene il percorso della location inesistente
		/// </summary>
		public string Location
		{
			get{return location;}
		}

		#region ISerializable Members

		/// <summary>
		/// Funzione per la serializzazione
		/// </summary>
		/// <param name="info">info</param>
		/// <param name="context">context</param>
		public override void GetObjectData(SerializationInfo info, StreamingContext context)
		{
			base.GetObjectData(info,context);
			info.AddValue("location",location);
		}

		#endregion
	}

	/// <summary>
	/// Eccezione utilizzata per segnalare l'accesso ad un elemento inesistente
	/// </summary>
	[Serializable]
	public class ElementNotExistsException : TeamProjectException
	{
		/// <summary>
		/// Nome dell'elemento, comprensivo di percorso
		/// </summary>
		protected string element;
		
		/// <summary>
		/// Costruttore utilizzato per la serializzazione
		/// </summary>
		/// <param name="info">info</param>
		/// <param name="context">context</param>
		protected ElementNotExistsException(SerializationInfo info, StreamingContext context):
		base(info,context)
		{
			element = (string)info.GetValue("element",typeof(string));
		}

		/// <summary>
		/// Costruisce un'istanza dell'eccezione
		/// </summary>
		/// <param name="element">Nome e percorso dell'elemento non trovato</param>
		public ElementNotExistsException(string element):
			this(element,element + " does not exists")
		{}

		/// <summary>
		/// Costruisce un'istanza dell'eccezione
		/// </summary>
		/// <param name="element">Nome e percorso dell'elemento non trovato</param>
		/// <param name="message">Messaggio d'errore ulteriore</param>
		public ElementNotExistsException(string element, string message):
			this(element,message,null)
		{}

		/// <summary>
		/// Costruisce un'istanza dell'eccezione
		/// </summary>
		/// <param name="element">Nome e percorso dell'elemento non trovato</param>
		/// <param name="message">Messaggio d'errore ulteriore</param>
		/// <param name="inner">Eccezione interna</param>
		public ElementNotExistsException(string element, string message, Exception inner)
			:base(message,inner)
		{
			this.element = element;
		}

		/// <summary>
		/// Nome dell'elemento. Pu contenere anche il percorso
		/// </summary>
		public string ElementName
		{
			get{return element;}
		}

		#region ISerializable Members

		/// <summary>
		/// Funzione per la serializzazione
		/// </summary>
		/// <param name="info">info</param>
		/// <param name="context">context</param>
		public override void GetObjectData(SerializationInfo info, StreamingContext context)
		{
			base.GetObjectData(info,context);
			info.AddValue("element",element);
		}

		#endregion
	}

	/// <summary>
	/// Eccezione che segnala la cancellazione di una location non vuota
	/// </summary>
	[Serializable]
	public class LocationNotEmptyException : TeamProjectException
	{
		/// <summary>
		/// Percorso della location che si  cercato di cancellare
		/// </summary>
		protected string location;
		/// <summary>
		/// Costruttore per la serializzazione
		/// </summary>
		/// <param name="info"></param>
		/// <param name="context"></param>
		protected LocationNotEmptyException(SerializationInfo info, StreamingContext context):
			base(info,context)
		{
			location = (string)info.GetValue("location",typeof(string));
		}

		/// <summary>
		/// Costruisce un'istanza dell'eccezione
		/// </summary>
		/// <param name="location">Percorso della location</param>
		public LocationNotEmptyException(string location):
			this(location,location + " is not empty")
		{}

		/// <summary>
		/// Costruisce un'istanza dell'eccezione
		/// </summary>
		/// <param name="location">Percorso della location</param>
		/// <param name="message">Messaggio ulteriore</param>
		public LocationNotEmptyException(string location, string message):
			this(location,message,null)
		{}

		/// <summary>
		/// Costruisce un'istanza dell'eccezione
		/// </summary>
		/// <param name="location">Percorso della location</param>
		/// <param name="message">Messaggio ulteriore</param>
		/// <param name="inner">Eccezione interna</param>
		public LocationNotEmptyException(string location, string message, Exception inner)
			:base(message,inner)
		{
			this.location = location;
		}

		/// <summary>
		/// Percorso della location che si intendeva distruggere
		/// </summary>
		public string Location
		{
			get{return location;}
		}

		#region ISerializable Members
		/// <summary>
		/// Funzione er la seializzazione
		/// </summary>
		/// <param name="info"></param>
		/// <param name="context"></param>
		public override void GetObjectData(SerializationInfo info, StreamingContext context)
		{
			base.GetObjectData(info,context);
			info.AddValue("location",location);
		}

		#endregion
	}
}
